#  expire.pl
#
#  Process a list of HTML files, removing all text within special
#  EXPIRE comments if the expiration date has passed.
#
#  Example:
#
#    foo <!-- EXPIRE 1941-12-05 --> bar <!-- /EXPIRE --> bug
#
#  would become 'foo  bar  bug' after 5 December 1941. The date must
#  be in ISO format. Also, the comments must appear exactly as
#  they are above, including spaces inside the comment.
#
#  We only rewrite the file if an EXPIRE comment expires.
#
#  EXPIRE comments can be safely nested iff the enclosing EXPIRE
#  expires after the enclosed EXPIRE.
#
#  Input: a list of files to process.
#
#  Mark L. Irons
#  25 November 2001


#--------------------------------------------------------------------
#  Get current date and convert it to ISO format

($s,$m,$h,$curday,$curmon,$curyear,$w,$y,$d) = localtime;
$curyear=$curyear+1900;                       # living in 21st century
$curmon=$curmon+1;                            # Jan is 1st month, not 0th
if ($curmon < 10) { $curmon = "0".$curmon; }  # pad month to two digits
if ($curday < 10) { $curday = "0".$curday; }  # pad day to two digits
$curdate = $curyear."-".$curmon."-".$curday;  # put it all together


#--------------------------------------------------------------------
#  Set patterns to search on:
#
#
#   <!-- EXPIRE yyyy-mm-dd --> ... <!-- /EXPIRE -->
#   |------------------------|     |--------------|
#          $openEXPIRE               $closeEXPIRE
#
#  $1 becomes an ISO date on match of $openEXPIRE

$openEXPIRE  = '<!-- EXPIRE (\d\d\d\d-\d\d-\d\d) -->';
$closeEXPIRE = '<!-- /EXPIRE -->';


#--------------------------------------------------------------------
#  Loop over files, processing each.

FILE:
while (<>) {
  $filename = $_;
  if (!open(HTMLFILE,$filename)) {
    chop $filename;
    warn "Can't open $filename, skipping: $!\n";
    next;
  }
  $modified = 0;      # boolean flag, true iff we expire anything
  $input    = "";     # text of input file
  $output   = "";     # text of modified file


  # Read the whole file into one string

  while (<HTMLFILE>) {
    $input .= $_;
  }
  close(HTMLFILE);


  # Search for open/close EXPIRE comments, and remove
  # everything between them if the expiration date has passed.

  while ($input =~ /$openEXPIRE/) {

    $output .= $`;

    # Check the expiration date. If it hasn't passed, then output the
    # comment and check the rest of the input.

    if ($1 gt $curdate) {     # check whether EXPIRE comment has expired
      $output .= $&;          # if not, output EXPIRE comment
      $input = $';
    }

    # Look for the matching close comment. If not found, print an error and
    # continue with the next file. Otherwise, discard everything between the
    # expiring comments and continue.

    else {
      $input = $';
      if ($input !~ /$closeEXPIRE/) {
        chop $filename;
        print "Error: $filename is missing an $closeEXPIRE comment\n";
        next FILE;
      }
      else {
        $modified = 1;
        $input = $';
      }
    }
  }
  $output .= $input;


  # write file back out only if it's been changed

  if ($modified) {
    if (open(HTMLFILE,"> ".$filename)) {
      print HTMLFILE $output;
      close HTMLFILE;
      print "Expired content from $filename";
    }
  }

} # all files
